<?php


/**
 * This is the model class for table "statstable".
 *
 * The followings are the available columns in table 'gatetable':
 * @property integer $id                Chiave primaria record
 * @property integer $Gate              seriale del varco
 * @property string  $Classe            classe di veicolo corrispondente allla statistica
 * @property string  $StartTime         ora inizio statistica
 * @property string  $EndTime           ora fine statistica
 * @property string  $StartDate         data inizio statistica
 * @property string  $EndDate           data fine statistica
 * @property string  $StartDateTime     data/ora inizio statistica
 * @property string  $EndDateTime       data/ora fine statistica
 * @property integer $lenghtTimeSec     durata in secondi dell'intervallo di statistica
 * @property integer $Counter           numero di veicoli
 * @property double  $MeanSpeed         velocità media dei veicoli
 * @property integer $SpeedLimit        limite di velocità
 * @property double  $Mean            velocità media dei veicoli
 * @property double  $Variance          varianza della velocità
 * @property integer $GroupID         id del gruppo di statistiche (statistiche dello stesso periodo dello stesso dispositivo, con classi differenti)
 * @property string  $Comments        commento
 * @property string  $epoch         data/ora inserimento record
 */

class VehiclesFrequencyByType extends ActiveRecord
{

  public static function model($className=__CLASS__)
  {
    return parent::model($className);
  }

  public function tableName()
  {
    return 'statstable';
  }

  public function relations()
  {
    return array(
      'speeds' => array(self::HAS_MANY, 'VehiclesFrequencyBySpeed', 'StatID'),
      'myGate'    => array(self::BELONGS_TO , 'Gate' , array('Gate'      => 'Gate')),
    );
  }

  public function behaviors()
  {
    return array('edatetimebehavior' => array('class' => 'ext.EDateTimeBehavior'));
  }

  public function byGate($gates)
  {
    if(!is_array($gates))
      $gates = array($gates);

    $this->getDbCriteria()->addInCondition('t.Gate', $gates);

    return $this;
  }

  public function byVehicleClass($vehiclesClass)
  {
    if (!is_array($vehiclesClass))
      $vehiclesClass = array($vehiclesClass);

    $this->getDbCriteria()->addInCondition('Classe', $vehiclesClass);
    return $this;
  }

  public function startingFrom(DateTime $startDateTime)
  {
    $this->getDbCriteria()->mergeWith(
      array(
        'condition' => 'StartDateTime >= :start',
        'params' => array(
          ':start' => DateHelper::formatISO9075DateTime($startDateTime),
        ),
      )
    );
    return $this;
  }

  public function endingAt(DateTime $endDateTime)
  {
    $this->getDbCriteria()->mergeWith(
      array(
        'condition' => 'EndDateTime <= :end',
        'params' => array(
          ':end' => DateHelper::formatISO9075DateTime($endDateTime),
        ),
      )
    );
    return $this;
  }

  public function groupByGate()
  {
  	$this->getDbCriteria()->mergeWith(
  			array('group' => 't.Gate')
  	);
  	
  	return $this;
  }
  public function groupByVehicleClass()
  {
  	$this->getDbCriteria()->mergeWith(
  			array('group' => 'Classe')
  	);
  	
  	return $this;
  }
  
  public function withGates($useJoin = false)
  {
  	$crit = new CDbCriteria();
  	if($useJoin)
  		$crit->join = "LEFT JOIN GateTable as myGate ON myGate.Gate = t.Gate";
  	else
  	{
  		$crit->with = array('myGate');
  		$this->getDbCriteria()->together = true;
  	}  		
  	
  	$this->getDbCriteria()->mergeWith($crit);
  	

  	 
  	return $this;
  }

  public function startingFromTime(DateTime $startTime)
  {
    $this->getDbCriteria()->mergeWith(
        array(
            'condition' => 'TIME(StartTime) >= :timestart',
            'params' => array(
                ':timestart' => DateHelper::formatISO9075Time($startTime),
            ),
        )
    );
    return $this;
  }
  
  public function endingAtTime(DateTime $endTime)
  {
    $this->getDbCriteria()->mergeWith(
        array(
            'condition' => 'TIME(StartTime) <= :timeend',
            'params' => array(
                ':timeend' => DateHelper::formatISO9075Time($endTime),
            ),
        )
    );
    return $this;
  }
  
}
